package gov.va.genisis2.ts.controller;

import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.impl.HttpSolrClient;
import org.owasp.esapi.ESAPI;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;

import com.fasterxml.jackson.databind.ObjectMapper;

import gov.va.genisis2.ts.common.dto.BookmarkDTO;
import gov.va.genisis2.ts.common.dto.ConceptMappingDTO;
import gov.va.genisis2.ts.common.dto.DataElementDTO;
import gov.va.genisis2.ts.common.dto.LabelDTO;
import gov.va.genisis2.ts.common.dto.ResponseWrapper;
import gov.va.genisis2.ts.common.dto.TripleDTO;
import gov.va.genisis2.ts.common.dto.UpdateResponseDTO;
import gov.va.genisis2.ts.common.dto.UpdateTripleDTO;
import gov.va.genisis2.ts.common.enums.ErrorEnum;
import gov.va.genisis2.ts.common.exception.TSDuplicateDataException;
import gov.va.genisis2.ts.common.exception.TSInernalSystemException;
import gov.va.genisis2.ts.common.exception.TSNoDataFoundException;
import gov.va.genisis2.ts.dto.ConceptCardDTO;
import gov.va.genisis2.ts.dto.SolrDTO;
import gov.va.genisis2.ts.service.IConceptCardService;
import gov.va.genisis2.ts.service.IDataElementService;
import gov.va.genisis2.ts.service.IMappingService;
import gov.va.genisis2.ts.service.ISolrService;
import gov.va.genisis2.ts.service.IUpdateTripleService;
import gov.va.genisis2.ts.utils.TSPropertiesUtil;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;

/**
 * REST controller class for Terminology Service
 * 
 * @author PII
 * @author PII
 *
 */
@CrossOrigin(origins = "*")
@RestController
@RequestMapping("/")
public class TermServiceController {

	private static final Logger LOGGER = LogManager.getLogger(TermServiceController.class);

	private final RestTemplate restTemplate = new RestTemplate();
	private HttpHeaders jsonHeaders;

	@Autowired
	private IConceptCardService conceptCardService;

	@Autowired
	private IUpdateTripleService updateTripleService;

	@Autowired
	private ISolrService solrService;

	@Autowired
	private IMappingService mappingService;

	@Autowired
	private TSPropertiesUtil propsUtil;
	
	@Autowired
	private IDataElementService dataElementService;

	@RequestMapping(value = "/searches", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Gets the concetps from solr by a search phrase", notes = "Returns the concepts from solr that matches the provided search phrase", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Concept API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> getConceptsOnSearchPhrase(@RequestParam(value = "searchPhrase", required = true) String searchPhrase, @RequestParam(defaultValue = "0") String start, @RequestParam(defaultValue = "20") String rows) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getConceptsOnSearchPhrase");
		}

		final String uri = String.format(propsUtil.getSolrSearchServiceEndpoint(), start, rows, searchPhrase);

		HttpEntity<String> httpEntity = new HttpEntity<String>(createJsonHeaders());

		ResponseEntity<String> solrResponseEntity;
		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			solrResponseEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, String.class);

			if (solrResponseEntity.getStatusCode() == HttpStatus.OK) {
				String solrResponseBody = solrResponseEntity.getBody();
				SolrDTO solrDto = new ObjectMapper().readValue(solrResponseBody, SolrDTO.class);
				conceptCardService.populateConceptName(solrDto);
				conceptCardService.populateBaselineSurveyIndex(solrDto);
				wrapper = new gov.va.genisis2.ts.common.dto.ResponseWrapper(solrDto);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.OK);
			}
		} catch (HttpClientErrorException e) {
			LOGGER.error("Error in fetching concept mappings " + e.getResponseBodyAsString(), e);
			wrapper = new ResponseWrapper(e.getResponseBodyAsString());
			responseEntityWithWrapper = createReposneEntity(wrapper, e.getStatusCode());
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		return responseEntityWithWrapper;
	}

	@RequestMapping(value = "/concept-cards", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Gets the concetp card details by concept URI", notes = "Returns the concept card details for a given concept URI", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Concept API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> getConceptCardsOnConceptUri(@RequestParam(value = "conceptUri", required = true) String conceptUri) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getConceptCardsOnConceptUri");
		}

		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			String decodedConceptUri = java.net.URLDecoder.decode(conceptUri, "UTF-8");

			ConceptCardDTO conceptCardDto = conceptCardService.getConceptCardsOnConceptUri(decodedConceptUri);

			if (null != conceptCardDto) {
				wrapper = new ResponseWrapper(conceptCardDto);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.OK);
			}
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		return responseEntityWithWrapper;
	}

	@RequestMapping(value = "/bookmarks", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Create Bookmark", notes = "Add a bookmarked concept", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Bookmarks API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> createBookmark(@RequestBody BookmarkDTO bookmarkDto) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("addBookmark");
		}
		HttpEntity<BookmarkDTO> httpEntity = new HttpEntity<BookmarkDTO>(bookmarkDto, createJsonHeaders());

		ResponseEntity<BookmarkDTO> responseEntity = null;
		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			// TerminologyServiceConstants.BookmarkingServiceUri + "/bookmarks"
			responseEntity = restTemplate.exchange(propsUtil.getMsCreateBookmarksUri(), HttpMethod.POST, httpEntity, BookmarkDTO.class);

			if (responseEntity.getStatusCode() == HttpStatus.CREATED) {
				BookmarkDTO returnedDto = (BookmarkDTO) responseEntity.getBody();
				wrapper = new ResponseWrapper(returnedDto);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.CREATED);
			}
		} catch (HttpClientErrorException e) {
			LOGGER.error("Error in creating bookmark " + e.getResponseBodyAsString(), e);
			wrapper = new ResponseWrapper(e.getResponseBodyAsString());
			responseEntityWithWrapper = createReposneEntity(wrapper, e.getStatusCode());
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		return responseEntityWithWrapper;
	}

	@RequestMapping(value = "/bookmarks/{username:.+}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Fetch Bookmark and Label By Username", notes = "Retruns the list of bookmarks label for a given Username", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Bookmarks API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> retreiveBookmarksByUsername(@PathVariable String username) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getBookmarksOnUserName");
		}
		HttpEntity<BookmarkDTO> httpEntity = new HttpEntity<BookmarkDTO>(createJsonHeaders());

		ResponseEntity<List<BookmarkDTO>> responseEntity = null;
		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			responseEntity = restTemplate.exchange(propsUtil.getMsGetBookmarksByUsernameUri(), HttpMethod.GET, httpEntity, new ParameterizedTypeReference<List<BookmarkDTO>>() {
			}, username);
			if (responseEntity.getStatusCode() == HttpStatus.OK) {
				List<BookmarkDTO> bookmarkDto = (List<BookmarkDTO>) responseEntity.getBody();

				for (BookmarkDTO b : bookmarkDto) {
					b.setHasMapping(mappingService.hasMapping(b.getConceptUri()));
				}
				wrapper = new ResponseWrapper(bookmarkDto);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.OK);
			}
		} catch (HttpClientErrorException e) {
			LOGGER.error("Error in fetching bookmarks for username " + e.getResponseBodyAsString(), e);
			wrapper = new ResponseWrapper(e.getResponseBodyAsString());
			responseEntityWithWrapper = createReposneEntity(wrapper, e.getStatusCode());
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		return responseEntityWithWrapper;
	}

	@RequestMapping(value = "/bookmarks/labels/{label}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Fetch Bookmark by label", notes = "Retruns the list of bookmarks for a given label", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Bookmarks API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> retreiveBookmarksByLabel(@PathVariable String label) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("retreiveBookmarksByLabel");
		}
		HttpEntity<BookmarkDTO> httpEntity = new HttpEntity<BookmarkDTO>(createJsonHeaders());

		ResponseEntity<List<BookmarkDTO>> responseEntity = null;
		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			responseEntity = restTemplate.exchange(propsUtil.getMsGetBookmarksByLabelUri(), HttpMethod.GET, httpEntity, new ParameterizedTypeReference<List<BookmarkDTO>>() {
			}, label);
			if (responseEntity.getStatusCode() == HttpStatus.OK) {
				List<BookmarkDTO> bookmarkDto = (List<BookmarkDTO>) responseEntity.getBody();
				wrapper = new ResponseWrapper(bookmarkDto);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.OK);
			}
		} catch (HttpClientErrorException e) {
			LOGGER.error("Error in fetching bookmarks by label " + e.getResponseBodyAsString(), e);
			wrapper = new ResponseWrapper(e.getResponseBodyAsString());
			responseEntityWithWrapper = createReposneEntity(wrapper, e.getStatusCode());
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		return responseEntityWithWrapper;
	}

	@RequestMapping(value = "/bookmarks/{id}", method = RequestMethod.DELETE, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "UnLabel Bookmarked Concept", notes = "Delete a label from a bookmark", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Bookmarks API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> deleteBookmark(@PathVariable int id) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("deleteBookmarksOnId");
		}

		HttpEntity<Integer> httpEntity = new HttpEntity<Integer>(createJsonHeaders());

		ResponseEntity<Integer> responseEntity = null;
		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			responseEntity = restTemplate.exchange(propsUtil.getMsDeleteBookmarksByIdUri(), HttpMethod.DELETE, httpEntity, Integer.class, id);

			if (responseEntity.getStatusCode() == HttpStatus.OK) {
				Integer returnedId = (Integer) responseEntity.getBody();
				wrapper = new ResponseWrapper(returnedId);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.OK);
			}
		} catch (HttpClientErrorException e) {
			LOGGER.error("Error in deleting bookmark " + e.getResponseBodyAsString(), e);
			wrapper = new ResponseWrapper(e.getResponseBodyAsString());
			responseEntityWithWrapper = createReposneEntity(wrapper, e.getStatusCode());
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		return responseEntityWithWrapper;
	}

	@RequestMapping(value = "/bookmarks/{id}/labels", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Label Bookmarked Concept", notes = "Add a label to a bookmark", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Bookmarks API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> labelBookmarkedConcept(@PathVariable int id, @RequestBody LabelDTO labelDto) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("addLabelToBookmark");
		}

		HttpEntity<LabelDTO> httpEntity = new HttpEntity<LabelDTO>(labelDto, createJsonHeaders());

		ResponseEntity<BookmarkDTO> responseEntity = null;
		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			responseEntity = restTemplate.exchange(propsUtil.getMsCreateBookmarksWithLabelUri(), HttpMethod.POST, httpEntity, BookmarkDTO.class, id);

			if (responseEntity.getStatusCode() == HttpStatus.CREATED) {
				BookmarkDTO returnedDto = (BookmarkDTO) responseEntity.getBody();
				wrapper = new ResponseWrapper(returnedDto);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.CREATED);
			}
		} catch (HttpClientErrorException e) {
			LOGGER.error("Error in adding label to bookmark", e);
			wrapper = new ResponseWrapper("Error in adding label to bookmark");
			responseEntityWithWrapper = createReposneEntity(wrapper, e.getStatusCode());
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		return responseEntityWithWrapper;
	}

	@RequestMapping(value = "/bookmarks/{id}/labels", method = RequestMethod.DELETE, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Label Bookmarked Concept", notes = "Add a label to a bookmark", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Bookmarks API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> unLabelBookmarkedConcept(@PathVariable int id, @RequestParam(value = "labelName", required = true) String labelName) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("removeLabelFromBookmark");
		}
		HttpEntity<?> httpEntity = new HttpEntity<>(createJsonHeaders());

		ResponseEntity<BookmarkDTO> responseEntity = null;
		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			responseEntity = restTemplate.exchange(propsUtil.getMsDeleteBookmarksByLabelUri(), HttpMethod.DELETE, httpEntity, BookmarkDTO.class, id, labelName);

			if (responseEntity.getStatusCode() == HttpStatus.OK) {
				BookmarkDTO bookmarkDto = (BookmarkDTO) responseEntity.getBody();
				wrapper = new ResponseWrapper(bookmarkDto);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.OK);
			}
		} catch (HttpClientErrorException e) {
			LOGGER.error("Error in remvoing label from bookmark " + e.getResponseBodyAsString(), e);
			wrapper = new ResponseWrapper(e.getResponseBodyAsString());
			responseEntityWithWrapper = createReposneEntity(wrapper, e.getStatusCode());
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		return responseEntityWithWrapper;
	}

	@RequestMapping(value = "/labels/{username:.+}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Get lables by Username", notes = "Retrieve lables by Username", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Bookmarks API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> retreiveLabelsByUsername(@PathVariable String username) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("retreiveLabelsByUsername");
		}
		HttpEntity<?> httpEntity = new HttpEntity<>(createJsonHeaders());

		ResponseEntity<List<LabelDTO>> responseEntity = null;
		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			responseEntity = restTemplate.exchange(propsUtil.getMsGetBookmarksLabelByUsernameUri(), HttpMethod.GET, httpEntity, new ParameterizedTypeReference<List<LabelDTO>>() {
			}, username);
			if (responseEntity.getStatusCode() == HttpStatus.OK) {
				List<LabelDTO> labelDto = (List<LabelDTO>) responseEntity.getBody();
				wrapper = new ResponseWrapper(labelDto);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.OK);
			}
		} catch (HttpClientErrorException e) {
			LOGGER.error("Error in fetching labels for username " + e.getResponseBodyAsString(), e);
			wrapper = new ResponseWrapper(e.getResponseBodyAsString());
			responseEntityWithWrapper = createReposneEntity(wrapper, e.getStatusCode());
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		return responseEntityWithWrapper;
	}

	@RequestMapping(value = "/concept-history-stack-top", method = RequestMethod.DELETE, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Pops the history for the concept card", notes = "Pops the stack of Concept URIs associated with the concept card, and returns the new URI", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Concept API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<String> popConceptCardHistory() {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("popConceptCardHistory");
		}

		String json = "{\"response\":null,\"message\":null}";
		return new ResponseEntity<String>(json, HttpStatus.OK);
	}

	@RequestMapping(value = "/mappings", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Fetch Concept Mappings by Concept-URI", notes = "Returns connect mapping details by conceptUri", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Mapping API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> getConceptMappingByConceptUriWithResponseWrapper(@RequestParam(value = "conceptUri", required = true) String conceptUri) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getConceptMappingByConceptUriWithResponseWrapper");
		}

		HttpEntity<ConceptMappingDTO> httpEntity = new HttpEntity<ConceptMappingDTO>(createJsonHeaders());

		ResponseEntity<ConceptMappingDTO> responseEntity = null;
		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			responseEntity = restTemplate.exchange(propsUtil.getMsGetConceptMappingUriByConceptUri(), HttpMethod.GET, httpEntity, ConceptMappingDTO.class, conceptUri);
			if (responseEntity.getStatusCode() == HttpStatus.OK) {
				ConceptMappingDTO conceptMappingDto = (ConceptMappingDTO) responseEntity.getBody();
				wrapper = new ResponseWrapper(conceptMappingDto);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.OK);
			}
		} catch (HttpClientErrorException e) {
			LOGGER.error("Error in fetching concept mappings " + e.getResponseBodyAsString(), e);
			wrapper = new ResponseWrapper(e.getResponseBodyAsString());
			responseEntityWithWrapper = createReposneEntity(wrapper, e.getStatusCode());
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
		}

		return responseEntityWithWrapper;
	}
	
	@RequestMapping(value = "/dataElements", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Fetch data elements", notes = "Returns all data elements in the Fuseki database", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Concept API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> getDataElements() {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getDataElements");
		}

		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			 List<DataElementDTO> dataElements = dataElementService.getDataElements();

			if (null != dataElements) {
				wrapper = new ResponseWrapper(dataElements);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.OK);
			}
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		return responseEntityWithWrapper;
		
	}

	@RequestMapping(value = "/mappings", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Creates Concept Mapping", notes = "Creates a new concept mapping", consumes = "application/json", produces = "application/json", response = String.class, tags = { "Mapping API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> createConceptMapping(@RequestBody ConceptMappingDTO conceptMappingDto) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("createConceptMapping");
		}

		return createReposneEntity(null, HttpStatus.METHOD_NOT_ALLOWED);
	}

	@RequestMapping(value = "/update", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Adds triples to Apache Jena/Fuseki", notes = "Adds triples to Apache Jena/Fuseki", consumes = "application/json", produces = "application/json", response = String.class, tags = { "CLI API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> updateTripleAdd(@RequestBody UpdateTripleDTO updateTripleDTO) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("updateTriples");
		}

		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			List<UpdateResponseDTO> reponseList = updateTripleService.addTriple(updateTripleDTO);

			if (null != reponseList) {
				wrapper = new ResponseWrapper(reponseList);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.OK);

				if (updateTripleDTO.isUpdateSolrIndex()) {
					// solr update
					List<TripleDTO> concepts = updateTripleDTO.getProperties();
					List<ConceptCardDTO> conceptCards = new ArrayList<ConceptCardDTO>();
					try (SolrClient solr = new HttpSolrClient(propsUtil.getSolrServiceEndpoint())) {

						for (TripleDTO concept : concepts) {

							// remove angle brackets
							String subject = concept.getS().replaceAll("\\<|\\>", "");
							conceptCards.add(conceptCardService.getConceptCardsOnConceptUri(subject));
						}
						solrService.addTripleToSolrDocument(conceptCards, solr);
					} catch (Exception e) {
						LOGGER.error(e.getMessage());
					}
				}
			}
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			wrapper = new ResponseWrapper(e.getMessage());
			responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.BAD_REQUEST);
		}

		return responseEntityWithWrapper;
	}

	@RequestMapping(value = "/update", method = RequestMethod.DELETE, consumes = "application/json", produces = "application/json")
	@ResponseBody
	@ApiOperation(value = "Removes triples to Apache Jena/Fuseki", notes = "Removes triples to Apache Jena/Fuseki", consumes = "application/json", produces = "application/json", response = String.class, tags = { "CLI API" })
	@ApiResponses({ @ApiResponse(code = 200, message = "Success", response = String.class) })
	public ResponseEntity<ResponseWrapper> updateTripleDelete(@RequestBody UpdateTripleDTO updateTripleDTO) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("updateTriples");
		}

		ResponseEntity<ResponseWrapper> responseEntityWithWrapper = null;
		ResponseWrapper wrapper = null;
		try {
			List<UpdateResponseDTO> reponseList = updateTripleService.deleteTriple(updateTripleDTO);

			if (null != reponseList) {
				wrapper = new ResponseWrapper(reponseList);
				responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.OK);

				if (updateTripleDTO.isUpdateSolrIndex()) {
					// solr update
					List<TripleDTO> concepts = updateTripleDTO.getProperties();
					List<String> conceptUris = new ArrayList<String>();
					try (SolrClient solr = new HttpSolrClient(propsUtil.getSolrServiceEndpoint())) {

						for (TripleDTO concept : concepts) {

							// remove angle brackets
							conceptUris.add(concept.getS().replaceAll("\\<|\\>", ""));
						}
						solrService.deleteTripleToSolrDocument(conceptUris, solr);
					} catch (Exception e) {
						LOGGER.error(e.getMessage());
					}
				}
			}
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			wrapper = new ResponseWrapper(e.getMessage());
			responseEntityWithWrapper = createReposneEntity(wrapper, HttpStatus.BAD_REQUEST);
		}

		return responseEntityWithWrapper;
	}

	/**
	 * No data found exception.
	 *
	 * @param ex
	 *            the ex
	 * @return the response entity
	 */
	@ExceptionHandler({ TSNoDataFoundException.class })
	public ResponseEntity<Object> exceptionHandler(TSNoDataFoundException ex) {
		return new ResponseEntity<Object>(
				ESAPI.encoder().encodeForHTML( ex.getMessage( )), HttpStatus.NOT_FOUND);
	}

	/**
	 * Duplicate data exception for create API calls
	 *
	 * @param ex
	 *            the ex
	 * @return the response entity
	 */
	@ExceptionHandler({ TSDuplicateDataException.class })
	public ResponseEntity<Object> exceptionHandler(TSDuplicateDataException ex) {
		return new ResponseEntity<Object>(
				ESAPI.encoder().encodeForHTML( ex.getMessage( )), HttpStatus.BAD_REQUEST);
	}

	/**
	 * Internal System Error exception
	 *
	 * @param ex
	 *            the ex
	 * @return the response entity
	 */
	@ExceptionHandler({ TSInernalSystemException.class })
	public ResponseEntity<Object> exceptionHandler(TSInernalSystemException ex) {
		return new ResponseEntity<Object>(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
	}

	private HttpHeaders createJsonHeaders() {
		if (jsonHeaders == null) {
			jsonHeaders = new HttpHeaders();
			jsonHeaders.setContentType(MediaType.APPLICATION_JSON);
			jsonHeaders.set("Accept-Encoding", "application/json");
		}

		return jsonHeaders;
	}

	/**
	 * Creates the response wrapper.
	 *
	 * @param wrapper
	 *            the wrapper
	 * @return the response entity
	 */
	private ResponseEntity<ResponseWrapper> createReposneEntity(ResponseWrapper wrapper, HttpStatus httpStatus) {
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Pragma", "No-cache");
		responseHeaders.add("Cache-Control", "no-cache");
		return new ResponseEntity<ResponseWrapper>(wrapper, responseHeaders, httpStatus);
	}
}
